"""Utilities for sending emails via the Gmail API and generating reminder content."""

from __future__ import annotations

import base64
import os
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from typing import Any, Iterable, Optional

# Try to import Google API client; if not available, we will fallback gracefully
try:
    from googleapiclient.discovery import build  # type: ignore
    from google_auth_oauthlib.flow import InstalledAppFlow  # type: ignore
    from google.oauth2.credentials import Credentials  # type: ignore
    from google.auth.transport.requests import Request  # type: ignore
except Exception:
    build = None
    InstalledAppFlow = None
    Credentials = None
    Request = None

# Gmail scopes needed for sending mail
SCOPES = ["https://www.googleapis.com/auth/gmail.send"]


def _ensure_iter(x: Optional[str | Iterable[str]]) -> list[str]:
    if not x:
        return []
    if isinstance(x, str):
        return [x]
    return [str(i) for i in x]


def _get_gmail_service() -> Optional[Any]:
    """
    Return an authorized Gmail API service or None if credentials are not configured.
    Looks for token.json / credentials.json in the project root by default.
    You may override paths by environment variables:
      - GMAIL_TOKEN_PATH
      - GMAIL_CREDENTIALS_PATH
    """
    if not (build and Credentials):
        return None

    token_path = os.getenv("GMAIL_TOKEN_PATH", "token.json")
    creds_path = os.getenv("GMAIL_CREDENTIALS_PATH", "credentials.json")

    creds = None
    if os.path.exists(token_path):
        try:
            creds = Credentials.from_authorized_user_file(token_path, SCOPES)
        except Exception:
            creds = None

    if creds and creds.expired and creds.refresh_token and Request:
        try:
            creds.refresh(Request())
        except Exception:
            creds = None

    if not creds:
        # Try interactive flow only if credentials.json exists
        if not os.path.exists(creds_path) or not InstalledAppFlow:
            return None
        try:
            flow = InstalledAppFlow.from_client_secrets_file(creds_path, SCOPES)
            creds = flow.run_local_server(port=0)
            # Persist token for future runs
            with open(token_path, "w", encoding="utf-8") as token:
                token.write(creds.to_json())
        except Exception:
            return None

    try:
        service = build("gmail", "v1", credentials=creds)
        return service
    except Exception:
        return None


def send_email(
    to: str | Iterable[str],
    subject: str,
    html_body: Optional[str] = None,
    text_body: Optional[str] = None,
    cc: Optional[str | Iterable[str]] = None,
    bcc: Optional[str | Iterable[str]] = None,
    sender: Optional[str] = None,
    **_: Any,  # absorb unexpected keyword args to stay compatible with callers
) -> bool:
    """
    Send an email via Gmail API.
    Returns True on success, False if sending failed or Gmail is not configured.

    Parameters:
        to: recipient or list of recipients
        subject: email subject
        html_body: HTML body (optional but recommended)
        text_body: plain text fallback (optional)
        cc, bcc: recipients (optional)
        sender: optional "From" header; Gmail will use the authorized account if omitted
    """
    to_list = _ensure_iter(to)
    cc_list = _ensure_iter(cc)
    bcc_list = _ensure_iter(bcc)

    if not to_list:
        return False

    # Build MIME message
    msg = MIMEMultipart("alternative")
    msg["To"] = ", ".join(to_list)
    if cc_list:
        msg["Cc"] = ", ".join(cc_list)
    if sender:
        msg["From"] = sender
    msg["Subject"] = subject

    # Attach bodies
    if text_body:
        msg.attach(MIMEText(text_body, "plain", "utf-8"))
    if html_body:
        msg.attach(MIMEText(html_body, "html", "utf-8"))
    if not text_body and not html_body:
        msg.attach(MIMEText("", "plain", "utf-8"))

    # Get Gmail service; if unavailable, fail gracefully
    service = _get_gmail_service()
    if service is None:
        # Not configured; indicate failure without raising
        return False

    try:
        raw = base64.urlsafe_b64encode(msg.as_bytes()).decode("utf-8")
        service.users().messages().send(userId="me", body={"raw": raw}).execute()
        return True
    except Exception:
        return False


def generate_reminder_html(rows: list[dict]) -> str:
    """
    Utility to render a simple HTML table of tasks for email bodies.
    Each row dict may contain: task_code, title, client_name, due_date, priority.
    """
    cells = []
    for r in rows:
        cells.append(
            "<tr>"
            f"<td>{r.get('task_code','')}</td>"
            f"<td>{r.get('title','')}</td>"
            f"<td>{r.get('client_name','')}</td>"
            f"<td>{r.get('due_date','')}</td>"
            f"<td>{r.get('priority','')}</td>"
            "</tr>"
        )
    table = (
        "<table border='1' cellspacing='0' cellpadding='6' style='border-collapse:collapse'>"
        "<thead><tr>"
        "<th>Task Code</th><th>Title</th><th>Client</th><th>Due Date</th><th>Priority</th>"
        "</tr></thead>"
        "<tbody>"
        + "".join(cells)
        + "</tbody></table>"
    )
    return table
